import Foundation
import SwiftUI
import Combine

@MainActor
class LifeClockViewModel: ObservableObject {
    @Published var currentLifeExpectancy: Double = 0.0
    @Published var lastUpdated: Date = Date()
    @Published var isLoading: Bool = false
    @Published var errorMessage: String?
    @Published var showPermissionRecoveryButton: Bool = false
    
    private let healthKitService = HealthKitService()
    private let predictionEngine = PredictionEngine.shared
    private var cancellables = Set<AnyCancellable>()
    
    init() {
        setupHealthKitAuthorization()
    }
    
    private func setupHealthKitAuthorization() {
        Task {
            do {
                print("LifeClockViewModel: Starting HealthKit authorization...")
                let healthData = try await healthKitService.initializeHealthKit()
                print("LifeClockViewModel: HealthKit authorization successful, updating life expectancy...")
                
                // If we got health data from initialization, use it directly
                if let healthData = healthData {
                    print("LifeClockViewModel: Using health data from initialization")
                    await updateLifeExpectancyWithData(healthData)
                } else {
                    print("LifeClockViewModel: No health data from initialization, fetching fresh data...")
                    await updateLifeExpectancy()
                }
            } catch {
                print("LifeClockViewModel: HealthKit authorization failed: \(error)")
                if let healthKitError = error as? HealthKitError {
                    switch healthKitError {
                    case .permissionsDenied(let deniedTypes):
                        let deniedList = deniedTypes.joined(separator: ", ")
                        errorMessage = "Essential health permissions were previously denied: \(deniedList)\n\nTo enable permissions:\n1. Open Health app\n2. Go to Privacy → Apps & Services\n3. Find 'LifeClock' and enable: \(deniedList)\n4. Return to this app and tap 'Retry'"
                        showPermissionRecoveryButton = true
                    default:
                        errorMessage = healthKitError.localizedDescription
                        showPermissionRecoveryButton = false
                    }
                } else {
                    errorMessage = "Failed to authorize HealthKit: \(error.localizedDescription)"
                    showPermissionRecoveryButton = false
                }
            }
        }
    }
    
    func updateLifeExpectancy() async {
        isLoading = true
        errorMessage = nil
        
        do {
            print("LifeClockViewModel: Checking HealthKit permissions...")
            
            // Check if we have required permissions
            if !healthKitService.hasRequiredPermissions() {
                let missingPermissions = healthKitService.getMissingPermissions()
                let deniedPermissions = healthKitService.getDeniedPermissions()
                print("LifeClockViewModel: Missing permissions: \(missingPermissions)")
                print("LifeClockViewModel: Denied permissions: \(deniedPermissions)")
                
                // If essential permissions were denied, show specific guidance
                if !deniedPermissions.isEmpty {
                    print("LifeClockViewModel: Essential permissions denied, showing guidance")
                    let deniedList = deniedPermissions.joined(separator: ", ")
                    errorMessage = "Essential health permissions were previously denied: \(deniedList)\n\nTo enable permissions:\n1. Open Health app\n2. Go to Privacy → Apps & Services\n3. Find 'LifeClock' and enable: \(deniedList)\n4. Return to this app and tap 'Retry'"
                    showPermissionRecoveryButton = true
                    isLoading = false
                    return
                } else if !missingPermissions.isEmpty {
                    // Filter to show only essential missing permissions
                    let essentialMissing = missingPermissions.filter { permission in
                        ["Resting Heart Rate", "Step Count", "Biological Sex", "Date of Birth"].contains(permission)
                    }
                    
                    if !essentialMissing.isEmpty {
                        errorMessage = "Missing essential health permissions: \(essentialMissing.joined(separator: ", ")). Please check Health app settings."
                    } else {
                        // Only optional permissions are missing, continue with available data
                        print("LifeClockViewModel: Only optional permissions missing, continuing with available data")
                    }
                }
            }
            
            print("LifeClockViewModel: HealthKit permissions verified, fetching data...")
            
            // Fetch latest health data
            let healthSnapshot = try await healthKitService.fetchLatestHealthData()
            
            // Use the fetched health data
            await updateLifeExpectancyWithData(healthSnapshot)
            
        } catch {
            errorMessage = "Failed to update life expectancy: \(error.localizedDescription)"
            isLoading = false
        }
    }
    
    func updateLifeExpectancyWithData(_ healthData: HealthDataSnapshot) async {
        isLoading = true
        errorMessage = nil
        
        do {
            print("LifeClockViewModel: Using provided health data for calculation...")
            
            // Use the provided health data directly
            let healthSnapshot = healthData
            
            // Create health data for prediction
            let healthData = PredictionHealthData(
                restingHeartRate: healthSnapshot.restingHeartRate ?? 70.0,
                vo2Max: healthSnapshot.vo2Max ?? 40.0,
                bmi: healthSnapshot.bodyMassIndex ?? 25.0,
                averageSleepHours: healthSnapshot.sleepAnalysis?.totalSleepHours ?? 7.5,
                bloodPressureSystolic: nil,
                bloodPressureDiastolic: nil,
                cholesterolTotal: nil,
                glucoseLevel: nil
            )
            
            // Get survey data
            let surveyIntegrationService = SurveyIntegrationService()
            surveyIntegrationService.refreshData()
            let surveyData = surveyIntegrationService.currentSurveyData ?? SurveyData()
            
            // Extract demographics from health data
            guard let age = healthSnapshot.age,
                  let biologicalSex = healthSnapshot.biologicalSex else {
                errorMessage = "Age and gender information required from HealthKit"
                isLoading = false
                return
            }
            
            let gender: Gender = {
                switch biologicalSex {
                case .male: return .male
                case .female: return .female
                default: return .other
                }
            }()
            
            // Generate prediction
            let prediction = predictionEngine.calculateLifeExpectancy(
                age: age,
                gender: gender,
                healthData: healthData,
                surveyData: surveyData
            )
            
            currentLifeExpectancy = prediction.lifeExpectancy
            lastUpdated = Date()
            
        } catch {
            if let healthKitError = error as? HealthKitError {
                errorMessage = healthKitError.localizedDescription
            } else {
                errorMessage = "Failed to update prediction: \(error.localizedDescription)"
            }
        }
        
        isLoading = false
    }
    
    func formatLifeExpectancy() -> String {
        let years = Int(currentLifeExpectancy)
        return "\(years) years"
    }
    
    func formatLastUpdated() -> String {
        let formatter = RelativeDateTimeFormatter()
        formatter.unitsStyle = .short
        return formatter.localizedString(for: lastUpdated, relativeTo: Date())
    }
    
    /// Retry permission recovery after user has manually enabled permissions in Health app
    func retryPermissionRecovery() async {
        isLoading = true
        errorMessage = nil
        showPermissionRecoveryButton = false
        
        do {
            print("LifeClockViewModel: Attempting permission recovery...")
            let healthData = try await healthKitService.attemptPermissionRecovery()
            
            if let healthData = healthData {
                print("LifeClockViewModel: Permission recovery successful, updating life expectancy...")
                await updateLifeExpectancyWithData(healthData)
            } else {
                print("LifeClockViewModel: Permission recovery failed")
                errorMessage = "Permissions still not available. Please check Health app settings."
                showPermissionRecoveryButton = true
            }
        } catch {
            print("LifeClockViewModel: Permission recovery failed: \(error)")
            if let healthKitError = error as? HealthKitError {
                switch healthKitError {
                case .permissionsDenied(let deniedTypes):
                    let deniedList = deniedTypes.joined(separator: ", ")
                    errorMessage = "Essential health permissions still denied: \(deniedList)\n\nPlease ensure you have enabled these permissions in the Health app."
                    showPermissionRecoveryButton = true
                default:
                    errorMessage = healthKitError.localizedDescription
                }
            } else {
                errorMessage = "Failed to recover permissions: \(error.localizedDescription)"
                showPermissionRecoveryButton = true
            }
        }
        
        isLoading = false
    }
}